/*--------------------------------------------------------------------------*\

	FILE........: WAVE.CPP
	TYPE........: C++ Module
	AUTHOR......: David Rowe
	DATE CREATED: 21/1/97

	Functions for reading and writing RIFF Wave files.

	NOTE: dont add any mprintfs here as mes_init() may not be called
        by people who use the vpb_wave_xxx functions without calling
        vpb_open() first.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

							INCLUDES

\*--------------------------------------------------------------------------*/

#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <memory.h>
#include <math.h>

#include "apifunc.h"
#include "vpbapi.h"
#include "wavecpp.h"
#include "wobbly.h"

/*--------------------------------------------------------------------------*\

				DEFINES

\*--------------------------------------------------------------------------*/

#define	FS		8000	// sample rate		

// wave file format tags

#define	WAVE_LINEAR	1
#define	WAVE_ALAW	6	
#define	WAVE_MULAW	7
#define WAVE_OKIADPCM	16

/*--------------------------------------------------------------------------*\

				TYPEDEFS

\*--------------------------------------------------------------------------*/

/*
typedef unsigned short USHORT;
typedef unsigned long ULONG;
*/

#pragma pack(1)			// force member alignment to one byte

// RIFF-WAVE chunk

typedef struct {
	char   riff[4];
	long   lwave;		// length of wave segment in bytes	
	char   wavefmt[8];
} WAVE;

// format chunk

typedef struct {
	ULONG  lfmt;		// length of format segment		
	USHORT fmt;
	USHORT channels;
	long   srate;
	long   bytesec;
	short  blockalign;
	short  bits_sample;
	short  cbsize;
} FMT; 

// fact chunk (only required for non-WAVE_LINEAR files)

typedef struct {
	char   fact[4];
	ULONG  length;		// length of fact chunk after this long
	ULONG  ldata;		// length of data segment in bytes	
} FACT;

// data chunk

typedef struct {
	char   data[4];
	ULONG  ldata;		// length of data segment in bytes	
} DATA;

// AU file header format...

#define	AU_FILE_MAGIC ((unsigned long)0x2e736e64)
#define AU_ENCODING_MULAW	1
#define	AU_ENCODING_ALAW	27
#define AU_ENCODING_LINEAR	3
#define	AU_ENCODING_ADPCM	23

#pragma pack(1)

typedef	struct {
	unsigned long magic;	// ".snd"
	unsigned long hdrsize;	// at least 24 bytes
	unsigned long datasize; // size of audio sample
	unsigned long encoding; // data format
	unsigned long sampling; // sample rate
	unsigned long channels; // 1 or 2...	
} AU;

#pragma pack()

// state data required for wave files

enum { WAVE_FILE, AU_FILE };

typedef struct {
	int	file;
	AU      au;
	WAVE	wave;
	FMT	fmt;
	FACT	fact;
	DATA	data;
	FILE	*f;
	USHORT	bytes_sam;
	ULONG   bytes_left; // track bytes read in this wave data chunk
} WFILE;

void au_swap(AU *au);
void swapl(unsigned long *l);

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_open_write()

	AUTHOR......: David Rowe
	DATE CREATED: 21/1/97

	Opens a RIFF Wave file.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_wave_open_write(void **ppv, char filename[], int mode)
//  void	*ppv		new state variables
//  char   filename[];		filename of wave file	
{
	try {
		wave_open_write(ppv, filename, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_wave_open_write"));
	}
	
    return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_open_read()

	AUTHOR......: David Rowe
	DATE CREATED: 14/12/97

	Opens a RIFF Wave file for reading.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_wave_open_read(void **ppv, char filename[])
//  void   *ppv			new state variables
//  char   filename[];		filename of wave file	
{
	try {
		wave_open_read(ppv, filename);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_wave_open_read"));
	}
	
    return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: wave_open_write()

	AUTHOR......: David Rowe
	DATE CREATED: 21/1/97

	Internal version of vpv_wave_open_write(), wobblies caught
	by calling function.

\*--------------------------------------------------------------------------*/

void wave_open_write(void **ppv, char filename[], int mode)
//  void    *ppv		new state variables
//  char    filename[];		filename of wave file	
//  int	    mode;		compression mode
{
	WFILE   *w;		// wave file ptr			

	CheckNew(w = new WFILE);
	*ppv = w;
	w->file = WAVE_FILE;

	w->file = WAVE_FILE;
	w->f = fopen(filename,"wb");
	if (w->f == NULL) {
		delete w;
		throw Wobbly(VPBAPI_WAVE_CANT_OPEN_FILE);
	}

	// validate mode ------------------------------------------
        
	switch(mode) {
		case VPB_LINEAR:
			w->fmt.fmt = WAVE_LINEAR;
			w->bytes_sam = 2;
			w->fmt.bits_sample = 16;
		break;
		case VPB_ALAW:
			w->fmt.fmt = WAVE_ALAW;
			w->bytes_sam = 1;
			w->fmt.bits_sample = 8;
		break;
		case VPB_MULAW:
			w->fmt.fmt = WAVE_MULAW;
			w->bytes_sam = 1;
			w->fmt.bits_sample = 8;
		break;
	/*	case VPB_OKIADPCM:
			h->fmt = WAVE_OKIADPCM;
			w->bytes_sam = 1;
			h->bits_sample = 4;
		break;*/
		default:
			throw Wobbly(VPBAPI_WAVE_FORMAT_NOT_SUPPORTED);
		break;
	}

	// set up riff wave chunk ---------------------------------

	memcpy(w->wave.riff,"RIFF",4);
	int sizeof_header = sizeof(WAVE) + sizeof(FMT) + sizeof(DATA);
	if (w->fmt.fmt != WAVE_LINEAR)
		sizeof_header += sizeof(FACT);
	w->wave.lwave = sizeof_header-8;	// length of wave segment	
	memcpy(w->wave.wavefmt,"WAVEfmt ",8);

	// set up format chunk ------------------------------------

	w->fmt.lfmt		= 0x12;			// format header length	
	w->fmt.channels		= 1;			// mono					
	w->fmt.srate		= FS;			// Fs = 8 kHz			
	w->fmt.bytesec		= FS*w->bytes_sam;	// bytes/second			
	w->fmt.blockalign	= w->bytes_sam;
	w->fmt.cbsize		= 0;
	
	// set up  fact chunk -------------------------------------

	memcpy(w->fact.fact,"fact",4);
	w->fact.length = 4;
	w->fact.ldata = 0;

	// set up data chunk --------------------------------------

	memcpy(w->data.data,"data",4);
	w->data.ldata = 0;                              // length of data segmt	
	// write header to disk

	fwrite(&w->wave, sizeof(WAVE), 1, w->f);
	fwrite(&w->fmt, sizeof(FMT), 1, w->f);
	if (mode != VPB_LINEAR) 
		fwrite(&w->fact, sizeof(FACT), 1, w->f);
	fwrite(&w->data, sizeof(DATA), 1, w->f);
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: wave_open_read()

	AUTHOR......: David Rowe
	DATE CREATED: 14/12/97

	Internal version of vpb_wave_open_read(), wobblies trapped by
	calling functions.

\*--------------------------------------------------------------------------*/

void wave_open_read(void **ppv, char filename[])
//  void   *ppv			new state variables
//  char   filename[];		filename of wave file	
{
    WFILE   *w;			// wave file ptr		

	CheckNew(w = new WFILE);
	*ppv = (void*)w;
        w->file = WAVE_FILE;

	w->f = fopen(filename,"rb");
	if (w->f == NULL) {
		delete w;
		throw Wobbly(VPBAPI_WAVE_CANT_OPEN_FILE);
	}

	// read wave file header -------------------------------------------

        fread(&w->au, sizeof(AU), 1, w->f);
	au_swap(&w->au);

	if(w->au.magic == AU_FILE_MAGIC)
	{
		w->file = AU_FILE;
		w->fmt.fmt = 0xff;
		w->fmt.srate = w->au.sampling;
		w->fmt.channels = (unsigned short)(w->au.channels);

		// read and discard comment
		char tmp;
		unsigned int  i;
		for(i=0; i<w->au.hdrsize-sizeof(AU); i++) {
			fread(&tmp, 1, 1, w->f);
		}

		switch(w->au.encoding)
		{
		case AU_ENCODING_LINEAR:
			w->fmt.fmt = WAVE_LINEAR;
			w->fmt.bytesec = w->au.sampling * 2;
			break;
		case AU_ENCODING_MULAW:
			w->fmt.fmt = WAVE_MULAW;
			w->fmt.bytesec = w->au.sampling;
			break;
		case AU_ENCODING_ALAW:
			w->fmt.fmt = WAVE_ALAW;
			w->fmt.bytesec = w->au.sampling;
			break;
		case AU_ENCODING_ADPCM:
			w->fmt.fmt = WAVE_OKIADPCM;
			w->fmt.bytesec = w->au.sampling / 2;
			break;
		}
	}
	else
	{
		w->file = WAVE_FILE;

		// read in wave file header

		fseek(w->f, 0l, SEEK_SET);
		fread(&w->wave, sizeof(WAVE), 1, w->f);

		// read in length param of wave header
		fread(&w->fmt, sizeof(ULONG), 1, w->f);

		// now read rest of header
		fread(&w->fmt.fmt, sizeof(char), w->fmt.lfmt, w->f);

		// now look for data chunk, bypassing other chunks

		char  chunk_id[4];
		int   found_data = 0;
		ULONG fact_length;
		int   ret;

		do {			
			fread(&chunk_id, sizeof(char), 4, w->f);
			if (strncmp(chunk_id, "data", 4) == 0) {
				found_data = 1;
			}
			else {
				// move past and ignore this chunk
				fread(&fact_length, sizeof(ULONG), 1, w->f);
				ret = fseek(w->f, fact_length, SEEK_CUR);

				// if this assert fires means error in
				// wave routine
				assert(ret == 0);
			}
		} while (!found_data);				

		// set up w->data 

		fread(&w->data.ldata, sizeof(ULONG), 1, w->f);
		w->bytes_left = w->data.ldata;
	}

	w->bytes_sam = (USHORT)ceil((float)(w->fmt.bytesec/w->fmt.srate));
	if(w->fmt.channels>1){
		delete w;
		throw Wobbly(VPBAPI_WAVE_FORMAT_NOT_SUPPORTED);
	}
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_close_write

	AUTHOR......: David Rowe
	DATE CREATED: 22/1/97

	Closes a RIFF Wave file after writing.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_wave_close_write(void *wv)
//  void   *wv;			ptr to wave file			
{
	WFILE   *w = (WFILE*)wv;	// wave file ptr		

	fseek(w->f, 0l, SEEK_SET);

	fwrite(&w->wave, sizeof(WAVE), 1, w->f);
	fwrite(&w->fmt, sizeof(FMT), 1, w->f);
	if (w->fmt.fmt != WAVE_LINEAR) 
		fwrite(&w->fact, sizeof(FACT), 1, w->f);
	fwrite(&w->data, sizeof(DATA), 1, w->f);

	fclose(w->f);
	delete w;
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_close_read

	AUTHOR......: David Rowe
	DATE CREATED: 14/12/97

	Closes a RIFF Wave file after reading.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_wave_close_read(void *wv)
//  void   *wv;		ptr to wave file			
{
    WFILE   *w = (WFILE*)wv;	// wave file ptr		

    fclose(w->f);
    delete w;
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_write()

	AUTHOR......: David Rowe
	DATE CREATED: 22/1/97

	Writes a block of speech samples to a RIFF Wave file.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_wave_write(void *wv, char buf[], long n)
//  void   *wv;		ptr to wave file			
//  char  buf[];	buffer of chars to write		
//  long   n;		number of bytes to write
{
    WFILE   *w = (WFILE*)wv;	// wave file ptr			

    int write = fwrite(buf, sizeof(char), n, w->f);
    (w->wave.lwave) += n;
    (w->fact.ldata) += n;
    (w->data.ldata) += n;

	return(write);
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_read()

	AUTHOR......: David Rowe
	DATE CREATED: 14/12/97

	Reads a block of speech samples from a RIFF Wave file.  Returns
	the number of bytes read.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_wave_read(void *wv, char buf[], long n)
//  void   *wv;		ptr to wave file				
//  char  buf[];	buffer of chars to read		
//  long   n;		length of buffer in samples		
{
    WFILE   *w = (WFILE*)wv;	// wave file ptr			
    int     ret;
    int     bytes_to_read;

    if (w->file == WAVE_FILE) {

	    // only read to end of wave chunk, as some wave files have
	    // non-audio chuncks after audio data

	    if (w->bytes_left > (ULONG)n) {
		    bytes_to_read = (ULONG)n;
	    }
	    else {
		    bytes_to_read = w->bytes_left;
	    }
	    ret = fread(buf, sizeof(char), bytes_to_read, w->f);
	    w->bytes_left -= bytes_to_read;

    }
    else {
	    // au files

	    ret = fread(buf, sizeof(char), n, w->f);
    }

    /* DR 5/2/03 used for debugging wave file bugs
    int i;
    for(i=0; i<ret; i++) {
	    if (buf[i] != (char)0xff) {
		    printf("[%d] %02x %c\n", i, buf[i], buf[i]);
	    }
    }
    */

    return ret;
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_seek()

	AUTHOR......: John Kostogiannis 
	DATE CREATED: 9/1/98

	Moves wave pointer to a specified location in samples, with respect
	to the beginning of the file.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_wave_seek(void *wv, long offset)
//  void	*wv;		ptr to wave file	
//  long	offset;		offset from start of file in bytes
{
	WFILE   *w = (WFILE*)wv;	// wave file ptr	
	long	sizeof_header;

	switch(w->file)
		{
		case WAVE_FILE:
			sizeof_header = sizeof(WAVE) + sizeof(FMT) + 
				        sizeof(DATA);
			if (w->fmt.fmt != WAVE_LINEAR)
				sizeof_header += sizeof(FACT);
			break;
		case AU_FILE:
			sizeof_header = w->au.hdrsize;
		}
	
	return(fseek(w->f, sizeof_header+offset, SEEK_SET));
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_set_sample_rate()

	AUTHOR......: David Rowe 
	DATE CREATED: 9/6/98

	Changes the sample rate of a wave file.

\*--------------------------------------------------------------------------*/

void WINAPI vpb_wave_set_sample_rate(void *wv, unsigned short rate)
//  void	      *wv;		ptr to wave file		
//  unsigned short    r;		new sampling rate		
{
    WFILE   *w = (WFILE*)wv;		// wave file ptr

    w->au.sampling  = rate;
    w->fmt.srate    = rate;			// Fs			 
    w->fmt.bytesec  = rate*w->bytes_sam;	// bytes/second		
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: vpb_wave_get_mode

	AUTHOR......: David Rowe 
	DATE CREATED: 25/6/98

	Determines the compression mode of a wave file.

\*--------------------------------------------------------------------------*/

int WINAPI vpb_wave_get_mode(void *wv, unsigned short *mode)
//  void	    *wv;	ptr to wave file		
//  unsigned short  *mode	compression mode of wave file
{
	try {
		wave_get_mode(wv, mode);
	}

	catch(Wobbly w){
		return(RunTimeError(w,"vpb_wave_get_mode"));
	}
	
    return(VPB_OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: wave_get_mode

	AUTHOR......: David Rowe 
	DATE CREATED: 25/6/98

	Determines the compression mode of a wave file.

\*--------------------------------------------------------------------------*/

void wave_get_mode(void *wv, unsigned short *mode)
//  void	    *wv;		ptr to wave file		
//  unsigned short  *mode		compression mode of wave file
{
    WFILE   *w = (WFILE*)wv;		// wave file ptr

    switch(w->file)
    {
    case WAVE_FILE:
	switch(w->fmt.fmt) {
		case WAVE_LINEAR:
			*mode = VPB_LINEAR;
		break;
		case WAVE_ALAW:
			*mode = VPB_ALAW;
		break;
		case WAVE_MULAW:
			*mode = VPB_MULAW;
		break;
/*		case WAVE_OKIADPCM:
			*mode = VPB_OKIADPCM;*/
		break;
		default:
			throw Wobbly(VPBAPI_WAVE_FORMAT_NOT_SUPPORTED);
		break;
	}
   case AU_FILE:
	switch(w->au.encoding)
	{
	case AU_ENCODING_LINEAR:
		*mode = VPB_LINEAR;
		break;
	case AU_ENCODING_ALAW:
		*mode = VPB_ALAW;
		break;
	case AU_ENCODING_MULAW:
		*mode = VPB_MULAW;
		break;
	}	
   }
}

/*--------------------------------------------------------------------------*\

	FUNCTION....: wave_get_size

	AUTHOR......: David Rowe 
	DATE CREATED: 29/9/98

	Determines the size of a wave file in samples

\*--------------------------------------------------------------------------*/

void wave_get_size(void *wv, unsigned long *bytes)
//  void	   *wv;		ptr to wave file		
//  unsigned long  *bytes;	size of wave file in bytes
{
    WFILE   *w = (WFILE*)wv;	// wave file ptr
    assert(wv != NULL);

    switch(w->file)
    {
    case WAVE_FILE:
	*bytes = w->data.ldata;
	break;
    case AU_FILE:
	*bytes = w->au.datasize;
	break;
    default:
	assert(0);
   }
}

void au_swap(AU *au) {
	swapl(&au->magic);
	swapl(&au->hdrsize);
	swapl(&au->datasize);
	swapl(&au->encoding);
	swapl(&au->sampling);
	swapl(&au->channels);
}

void swapl(unsigned long *l) {
	unsigned char b0,b1,b2,b3;
	unsigned long swapped;

	b0 = (unsigned char)((*l>>24) & 0xff);
	b1 = (unsigned char)((*l>>16) & 0xff);
	b2 = (unsigned char)((*l>>8) & 0xff);
	b3 = (unsigned char)(*l & 0xff);
	
	swapped = (b3<<24) + (b2<<16) + (b1<<8) + b0;
	*l = swapped;
}
